% Controlling The Absorbing Boundary Layer Example
%
% This example demonstrates how to control the parameters of the PML within
% k-Wave via optional input parameters. It builds on the Homogeneous
% Propagation Medium, Heterogeneous Propagation Medium and Saving Movie
% Files examples.  
%
% author: Bradley Treeby
% date: 30th June 2009
% last update: 16th July 2009
%  
% This example is part of the k-Wave Toolbox (http://www.k-wave.org)

clear all;

% =========================================================================
% SIMULATION
% =========================================================================

% modify this parameter to run the different examples
example_number = 1;

% create the computational grid
Nx = 128;           % number of pixels in the x (column) direction
Nz = 128;           % number of pixels in the z (row) direction
dx = 0.1e-3;        % pixel width [m]
dz = 0.1e-3;        % pixel height [m]
kgrid = makeGrid(Nx, dx, Nz, dz);

% define the properties of the propagation medium
c = 1500;          % [m/s]
rho = 1000;        % [kg/m^3]

% create initial pressure distribution using makeDisc
disc_magnitude = 5;
disc_x_pos = 50;    % pixels
disc_z_pos = 50;    % pixels
disc_radius = 8;    % pixels
disc_1 = disc_magnitude*makeDisc(Nx, Nz, disc_x_pos, disc_z_pos, disc_radius);

disc_magnitude = 3;
disc_x_pos = 60;    % pixels
disc_z_pos = 80;    % pixels
disc_radius = 5;    % pixels
disc_2 = disc_magnitude*makeDisc(Nx, Nz, disc_x_pos, disc_z_pos, disc_radius);

p0 = disc_1 + disc_2;

% define a centered circular sensor
sensor_radius = 4e-3;   % [m]
num_sensor_points = 50;
sensor_mask = makeCartCircle(sensor_radius, num_sensor_points);

% set the input arguments
switch example_number
    case 1
        input_args = {'PMLAlpha', 0};
    case 2
        input_args = {'PMLSize', 5};
    case 3
        input_args = {'PMLInside', false};
end

% run the simulation
sensor_data = kspaceFirstOrder2D(p0, kgrid, c, rho, 'auto', sensor_mask, input_args{:});

% =========================================================================
% VISUALISATION
% =========================================================================

% plot the simulated sensor data
figure;
imagesc(sensor_data, [-1, 1]);
colormap(getColorMap);
ylabel('Sensor Position');
xlabel('Time Step');
colorbar;